########################################################################
# SERIAL ROUTINES
#
#    All of the routines that interface to the serial port are here.
########################################################################

########################################################################
# serialInit() -	Initialize the serial values
########################################################################
proc serialInit {} {
    global RS232
    global tcl_platform

    if { $tcl_platform(platform) == "windows" } {
	set RS232(device) [iniGetAttribute "RS232" "Windows-Device"]
	set RS232(speed) [iniGetAttribute "RS232" "Windows-Speed"]
    } else {
	set RS232(device) [iniGetAttribute "RS232" "Unix-Device"]
	set RS232(speed) [iniGetAttribute "RS232" "Unix-Speed"]
    }

    set RS232(fd)      0
}

########################################################################
# serialBreak() -	Generates a break on the serial line,
#                       which is assumed to be open.
#                       No errors are generated here...that is, if a
#                       break doesn't work, then no big deal.
########################################################################
proc serialBreak {} {
    global RS232

    if { ! [catch {flush $RS232(fd)}] } {
	catch {fconfigure $RS232(fd) -ttycontrol {BREAK 1}}
	after 500
	catch {fconfigure $RS232(fd) -ttycontrol {BREAK 0}}
    }
}

########################################################################
# Serial Configuration
#
#    The following strings represent the configuration of the serial
#    port for both UNIX and windows.
#    NOTE that the buffersize was chosen to allow a full track of data
#    to be stored for the largest format supported:
#         20 sectors * (256 + 1 + 2) = 20 * 259 = 5180 (plus some slop)
########################################################################

set serialConfigWindows "-buffering none \
	                 -buffersize 5200 \
			 -sysbuffer 5200 \
	                 -encoding binary \
			 -handshake none \
			 -translation binary \
			 -blocking 1 \
			 -timeout 100"

set serialConfigUnix "-buffering none \
	              -buffersize 5200 \
	              -encoding binary \
                      -handshake none \
		      -translation binary \
                      -blocking 1 \
		      -timeout 100"

########################################################################
# serialOpen() -	Opens the serial port and sets the baud rate.
#                       Raises an error if necessary.
#   NOTE - I may have accidentally changed UNIX blocking
########################################################################
proc serialOpen {} {
    global RS232
    global tcl_platform
    global serialConfigWindows
    global serialConfigUnix

    if { [catch {if { $tcl_platform(platform) == "windows" } {
	               set RS232(fd) [open $RS232(device) { "RDWR" }]
	        } else {
		       set RS232(fd) [open $RS232(device) { "RDWR" "NOCTTY" }]
	    }   }]} {
		tk_messageBox -default ok -icon error -title "ERROR"  \
			-type ok -message "Can't open $RS232(device), please change RS232 device."

		set RS232(fd) 0
		error "Bad Serial Device Open"
    }

    if { $tcl_platform(platform) == "windows" } {
	set config $serialConfigWindows
    } else {
	set config $serialConfigUnix
    }

    set config [concat $config -mode "$RS232(speed),n,8,1"]

    if { [catch {eval "fconfigure $RS232(fd) $config"}] } {
	    tk_messageBox -default ok -icon error -title "ERROR"  \
			-type ok -message "Bad configure of $RS232(device), please check RS232 device."
	    error "Can't fconfig serial device."
    }

    return
}

########################################################################
# proc serialClose() -	Closes the serial line.
#         
#                       No errors will come back from this routine.
#                       If a close is attempted on a non-open port,
#                       no big deal.
########################################################################
proc serialClose {} {
    global RS232

    catch {close $RS232(fd)}
    set RS232(fd) 0

}

########################################################################
# serialRead() -Read the specified amount from the SVD.  Will wait for
#               the given amount until "timout", after
#               which, this routine will return, potentially with only
#               part (or none) of the data.
#
#               It is assumed that the transmission is done in a fast
#               sequence of at least the amount that was specified in
#               this call.  In this way, the timeout amount of 100 ms will
#               catch all of the data.  NOTE - the normal firmware loop for
#               transmission of data leaves up to 10us between characters,
#               so 100 ms is long enough to ENSURE that we got everything.
#               NOTE - at 19200 baud, we get a character every 500 us.
#
#               Assumes that the RS232(fd) has been opened and configured.
#
#               It is assumed that data is ready when this routine is called.
#               Otherwise, this routine may return without data.
#
#               If the count is ZERO, then a read of all available is done.
########################################################################
proc serialRead {count} {
    global  RS232

    if { $count == 0 } {
	return [read $RS232(fd)]
    } else {
	return [read $RS2332(fd) $count]
    }
#    if { $count == 0 } {
#	catch [read $RS232(fd)] data
#    } else {
#	catch [read $RS232(fd) $count] data
#    }
#
#    return $data
}

########################################################################
# serialWrite() -	Writes the given data out to the serial port.
#
#               Assumes that the RS232(fd) has been opened and configured.
#
#               Upon error, a TCL_ERROR is raised so it can be caught
#               and the serial port re-initialized or something.
#               NOTE - the puts() is what raises the error condition.
########################################################################
proc serialWrite {data} {
    global RS232

    puts -nonewline $RS232(fd) $data
}

########################################################################
# serialConnected() -	Returns 1 (TRUE) if the serial port is connected.
#                       0 (FALSE) otherwise.
########################################################################
proc serialConnected {} {
    global  RS232

    if { $RS232(fd) == 0 } {
	return 0
    }
    return 1
}

########################################################################
# proc probeSerial() -	Probe the serial ports to see which ones are
#                       available.
########################################################################
proc probeSerialPorts {} {
    global RS232
    global availableSerialPorts
    global tcl_platform

    set availableSerialPorts {}

    if { $tcl_platform(platform) == "windows" } {

	if { [iniGetAttribute "Program" "AutoProbe"] == 0 } {
	    lappend availableSerialPorts "com1:"
	    lappend availableSerialPorts "com2:"
	    lappend availableSerialPorts "com3:"
	    lappend availableSerialPorts "com4:"
	    return
	}

	for {set i 1} { $i < 10 } {incr i} {
	    if { ! [catch {open "com$i:" "RDWR"} fd] } {
		lappend availableSerialPorts "com$i:"
		close $fd
	    }
	}
	if { [llength $availableSerialPorts] == 0 } {
	    tk_messageBox -default ok -icon error -title "ERROR"  \
		      -type ok -message "No serial ports found."
	    lappend availableSerialPorts "NONE FOUND"
	}
	set RS232(device) [lindex $availableSerialPorts 0]
    }
}

